# Hey Emacs, this is a -*- shell-script -*- !!!

# lsvpd-functions.bash - Shared support functions.

# This file is part of the Linux lsvpd package.

# (C) Copyright IBM Corp. 2002, 2003, 2004

# Maintained by  Martin Schwenke <martins@au.ibm.com>

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    
# $Id: lsvpd-functions.bash,v 1.1 2006/04/11 18:38:28 emunson Exp $

dec2hex ()
{
    printf "%x\n" "$1"
}

hex2dec ()
{
    # Using implicit arithmetic conversion is faster than printf?
    case "$1" in
	0x*) echo $((  $1)) ;;
	*)   echo $((0x$1)) ;;
    esac
}

# Doesn't invoke mkdir unless necessary, since it is an external program.
ensure_directory ()
{
    local d="$1"
    [ -d "$d" ] || mkdir -p "$d"
}

# Move directory contents from $tmp_dir to $dst_dir.  Every effort is
# made to destroy $tmp_dir.  Some subdirectories may already exist, so
# they are handled carefully.  We don't bother with files and
# subdirectories starting with '.'.
move_directory ()
{
    local tmp_dir="$1"
    local dst_dir="$2"

    # Basic sanity check.
    [ -d "$tmp_dir" ] || return

    if [ ! -d "$dst_dir" ] ; then
	ensure_directory "${dst_dir%/*}"  # dirname
	mv "$tmp_dir" "$dst_dir"
    else
	for i in "${tmp_dir}/"* ; do
	    b="${i##*/}"  # basename
	    d="${dst_dir}/${b}"

	    # If target doesn't exist then just move it.
	    if [ ! -e "$d" ] ; then
		debug_cmd mv "$i" "$d"
	    # If both source and target are existing directories then
	    # recurse.
	    elif [ -d "$d" -a -d "$i" ] ; then
		move_directory "$i" "$d"
	    fi

	    # Files that exist at both ends and file/directory
	    # mismatches are ignored.
	done
	debug_cmd rmdir "$tmp_dir"
    fi
}

process_db_path_options ()
{
    [ -n "$db_path_option" ] || db_path_option="d"

    db=""
    tgz=""

    # Using getopts here is just too hard.
    while [ -n "$1" ] ; do
	case "$1" in
	    -${db_path_option})
		shift ; db="$1" ; shift
		if [ -z "$db" ] ; then
		    echo "usage: -${db_path_option} option requires a directory path" 1>&2
		    exit 1
		fi
		if [ ! -d "$db" ] ; then
		    echo "no such directory: \"${db}\"" 1>&2
		    exit 1
		fi
		;;
	    -z)
		shift ; tgz="$1" ; shift
		if [ -z "$tgz" ] ; then
		    echo "usage: -z option requires an archive path" 1>&2
		    exit 1
		fi
		if [ ! -f "$tgz" ] ; then
		    echo "no such file: \"${tgz}\"" 1>&2
		    exit 1
		fi
		;;
	    (*) shift ;;
	esac
    done

    return 0
}

unpack_device_tree_archive ()
{
    local tgz="$1"

    local dbarchivedir="/tmp/lsvpd$$"
    trap "chmod -R +w $dbarchivedir >/dev/null 2>&1 ; rm -rf $dbarchivedir >/dev/null 2>&1" 0 1 2 3 13 15
    mkdir -p "$dbarchivedir"

    (cd "$dbarchivedir" && tar xzf -) <"$tgz" >/dev/null 2>&1
    db=$(echo "${dbarchivedir}/"*)
    # Check for system-id is for backward compatibility.
    if [ ! -d "${db}/bus" -a ! -f "${db}/system-id" ] ; then
	echo "\"$tgz\" is not a valid device-tree archive" 1>&2
	usage
    fi
}

sanitise ()
{
    echo "$@" | sed -e 's/^[ \t]*//' -e 's/[ \t]*$//'
}

hex2ascii ()
{
    local hex="$1"

    local i
    for i in $(echo $hex | sed -e 's/\(..\)/\1 /g') ; do
	printf "\x$i"
    done
}

hex2substring ()
{
    local hex="$1"
    local start="$2"
    local length="$3"

    local width=2

    # The echo causes leading/trailing whitespace to be stripped.
    # Also internal whitespace is compressed.
    echo $(hex2ascii ${hex:$(($width * $start)):$(($width * $length))})
}

######################################################################

# MULTIPLEXING:
#
# Doing:
#
#   make_multiplexed f
#
# will allow a calls like:
#
#   f yip args ...
#
# which will try to call
#
#   add_device_yip args ...
#
# if it exists.  If not, the next attempt will be
#
#   add_device_DEFAULT yip args ...
#
# If the DEFAULT function doesn't exist, failure is silent.

multiplex ()
{
    local func="$1" ; shift
    local type="$1" ; shift

    local f="${func}_${type}"
    if type -t "$f" >/dev/null && ! type -P "$f" >/dev/null ; then
	"$f" "$@"
    else
	f="${func}_DEFAULT"
	if type -t "$f" >/dev/null && ! type -P "$f" >/dev/null  ; then
	    "$f" "$type" "$@"
	else
	    : "No function defined for \"${func}\" \"${type}\""
	fi
    fi
}

make_multiplexed ()
{
    eval "$1 () { multiplex $1 \"\$@\" ; }"
}

######################################################################

do_setup ()
{
    local d i
    for d in "common" "$@" "common-post" ; do
	for i in "${LSVPD_LIBDIR}/${d}.d/"* ; do
	    . "$i"
	done
    done
}

######################################################################

# In general the 2nd loop can theoretically spin forever.  However,
# since we are the only ones manipulating the lockfile, it can't
# happen.
lock_file ()
{
    local file="$1"
    local timeout="${2:-10}"

    local lf="${file}.LOCK"

    # Get unique filename.
    local uf="${lf}.$$.$RANDOM"
    while [ -f "$uf" ] ; do
	uf="${lf}.$$.$RANDOM"
    done

    # Drop in PID.
    echo $$ 2>/dev/null >"$uf"

    # Sanity check.  dirname($file) might not exist.
    [ -f "$uf" ] || return 1

    local count pid mylock
    count=0
    mylock=false

    # Create hard link - ln is atomic.
    while [ $count -lt $timeout ] && \
	! { ln "$uf" "$lf" 2>/dev/null && mylock=true ; } ; do
        # Didn't get lock.
	read pid 2>/dev/null <"$lf"
	if [ -n "$pid" ] ; then
	    if kill -0 $pid 2>/dev/null ; then
		# PID still valid, sleep.
		sleep 1
		count=$(($count + 1))
	    else
		# PID invalid, stale lock, remove, spin.
		rm -f "$lf"
	    fi
	fi
	# If couldn't get $pid, lockfile must have gone, spin.
    done
    
    rm -f "$uf"
    $mylock
}

# Obviously unlock_file should only be called by someone who has the lock!
unlock_file ()
{
    local file="$1"

    local lf="${file}.LOCK"

    rm -f "$lf"
}
